/***************************************
  $Header: /home/amb/wwwoffle/RCS/spool.c 1.12 1997/08/03 09:24:18 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.2d.
  Handle all of the spooling of files in the spool directory.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <utime.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include "wwwoffle.h"
#include "errors.h"



/*++++++++++++++++++++++++++++++++++++++
  Open a file in the outgoing directory to write into / read from.

  int OpenOutgoingSpoolFile Returns a file descriptor, or -1 on failure.

  int rw Set to true to read, else false.
  ++++++++++++++++++++++++++++++++++++++*/

int OpenOutgoingSpoolFile(int rw)
{
 int fd=-1;
 int low=0,high=0;
 DIR *dir;
 struct dirent* ent;

 /* Open the outgoing directory. */

 dir=opendir("outgoing");
 if(!dir)
   {
    PrintMessage(Inform,"Cannot open outgoing directory [%!s]; creating one.");
    if(mkdir("outgoing",0755) || !(dir=opendir("outgoing")))
      {PrintMessage(Warning,"Cannot create or access outgoing directory [%!s].");return(-1);}
   }

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read outgoing directory [%!s].");closedir(dir);return(-1);}
 ent=readdir(dir);  /* skip .. */

 /* Go through the files and find the lowest/highest numbers. */

 while((ent=readdir(dir)))
   {
    int number=atoi(ent->d_name);

    if(number)
      {
       if(!low  || number<low)
          low=number;
       if(!high || number>high)
          high=number;
      }
   }

 closedir(dir);

 /* Open the lowest to read or a temporary one to write. */

 if(rw && low)
   {
    for(;low<=(high+MAX_SERVERS);low++) /* start at the lowest and keep going to the highest. */
      {
       struct stat buf;
       char name[24];

       sprintf(name,"outgoing/%d",low);

       if(stat(name,&buf))
          PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",name);
       else
          if((fd=open(name,O_RDONLY))==-1)
             PrintMessage(Inform,"Cannot open file '%s' [%!s]; race condition?",name);
          else
             if(unlink(name))
               {
                PrintMessage(Inform,"Cannot unlink file '%s' [%!s]; race condition?",name);
                close(fd); fd=-1;
               }
             else
                break;
      }
   }
 else if(!rw)
   {
    char name[24];

    sprintf(name,"outgoing/tmp.%d",getpid());

    fd=open(name,O_WRONLY|O_CREAT,0644);
    if(fd==-1)
       PrintMessage(Inform,"Cannot open file '%s' [%!s]",name);
   }

 return(fd);
}


/*++++++++++++++++++++++++++++++++++++++
  Close the temporary outgoing spool file and rename it.

  char *CloseOutgoingSpoolFile Return the name of the file in the outgoing directory.

  int fd The file descriptor that was being written to.
  ++++++++++++++++++++++++++++++++++++++*/

char *CloseOutgoingSpoolFile(int fd)
{
 int high=0,maxhigh;
 DIR *dir;
 struct dirent* ent;
 char tmpname[24];
 static char name[24];

 close(fd);

 /* Open the outgoing directory. */

 dir=opendir("outgoing");
 if(!dir)
   {
    PrintMessage(Inform,"Cannot open outgoing directory [%!s]; creating one.");
    if(mkdir("outgoing",0755) || !(dir=opendir("outgoing")))
      {PrintMessage(Warning,"Cannot create or access outgoing directory [%!s].");return(NULL);}
   }

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read outgoing directory [%!s].");closedir(dir);return(NULL);}
 ent=readdir(dir);  /* skip .. */

 /* Go through the files and find the highest number. */

 while((ent=readdir(dir)))
   {
    int number=atoi(ent->d_name);

    if(number)
       if(!high || number>high)
          high=number;
   }

 closedir(dir);

 /* Rename the temporary file to the new file. */

 sprintf(tmpname,"outgoing/tmp.%d",getpid());

 maxhigh=high;
 for(high++;high<=(maxhigh+4*MAX_SERVERS);high++) /* start 1 higher and keep going */
   {
    struct stat buf;

    sprintf(name,"outgoing/%d",high);

    if(!stat(name,&buf))
       PrintMessage(Inform,"Can stat file '%s'; race condition?",name);
    else
       if((fd=open(name,O_WRONLY|O_CREAT|O_EXCL,0644))==-1)
          PrintMessage(Inform,"Cannot open file '%s' [%!s]; race condition?",name);
       else
         {
          close(fd);
          if(rename(tmpname,name))
             PrintMessage(Inform,"Cannot rename file '%s' to '%s' [%!s].",tmpname,name);
          else
             break;
         }
   }

 return(&name[9]);
}


/*++++++++++++++++++++++++++++++++++++++
  Delete a specified file from the cache.

  int DeleteOutgoingSpoolFile Return 1 if the file from the cache was also deleted.

  char *name The name of the file to delete.

  char **url The name of the page that the file had requested.
  ++++++++++++++++++++++++++++++++++++++*/

int DeleteOutgoingSpoolFile(char *name,char **url)
{
 char *path=(char*)malloc(strlen(name)+16);
 FILE *file=NULL;
 int ret=0;

 strcpy(path,"outgoing/");
 strcat(path,name);

 file=fopen(path,"r");
 if(file)
   {
    char *req;

    req=ParseRequest(file,url);

    if(url)
      {
       char *host,*path,*args;
       int spool;

       SplitURL(*url,&host,&path,&args);
       spool=OpenWebpageSpoolFile(1,host,path,args);

       if(spool!=-1)
         {
          char reply[65],*head="HTTP/1.0 404 WWWOFFLE Will Get\r\n"; /* This line must not be changed (see messages.c). */

          read(spool,reply,64);
          close(spool);

          if(!strncmp(reply,head,strlen(head)))
            {
             ret=1;
             OpenWebpageSpoolFile(-1,host,path,args);
            }
         }
      }

    if(req)
       free(req);

    fclose(file);
   }

 if(unlink(path))
   PrintMessage(Warning,"Cannot unlink outgoing request '%s' [%!s].",name);

 return(ret);
}


/*++++++++++++++++++++++++++++++++++++++
  Open a file in a spool subdirectory to write into / read from / delete.

  int OpenWebpageSpoolFile Returns a file descriptor.

  int rw Set to 1 to read, 0 to write, -1 to delete, -2 to touch.

  char *host The hostname from the URL.

  char *path The pathname from the URL.

  char *args The arguments to the URL.
  ++++++++++++++++++++++++++++++++++++++*/

int OpenWebpageSpoolFile(int rw,char *host,char *path,char *args)
{
 struct stat buf;
 int fd=-1;
 char *file=(char*)malloc(strlen(host)+strlen(path)+16),*newpath=(char*)malloc(strlen(path)+2);
 unsigned long hash=0;
 int i;

 /* Modify the path and args. */

 if(!*path)
   {
    newpath[0]=PATH_SEP;
    newpath[1]=0;
   }
 else
   {
    for(i=0;path[i];i++)
       if(path[i]=='/')
          newpath[i]=PATH_SEP;
       else
          newpath[i]=path[i];
    newpath[i]=0;
   }

 if(args)
    hash=MakeHashFromArgs(args);

 if(args)
    sprintf(file,"%s/%s%c%c%08lx",host,newpath,PATH_SEP,PATH_SEP,hash);
 else
    sprintf(file,"%s/%s",host,newpath);

 /* Create or check for the directory for the spool file. */

 if(stat(host,&buf))
   {
    PrintMessage(Inform,"Directory '%s' does not exist [%!s]; creating one.",host);
    if(mkdir(host,0755))
      {PrintMessage(Warning,"Cannot create directory '%s' [%!s].",host);free(file);return(-1);}
   }
 else
    if(!S_ISDIR(buf.st_mode))
      {PrintMessage(Warning,"The file '%s' is not a directory.",host);free(file);return(-1);}

 /* Create the file for the web page. */

 if(rw==0)
    fd=open(file,O_RDWR|O_CREAT,0644);
 else if(rw==1)
    fd=open(file,O_RDONLY);
 else if(rw==-1)
    unlink(file);
 else if(rw==-2)
    utime(file,NULL);

 if(args && (rw<0 || fd!=-1))
   {
    sprintf(file,"%s/%c%c%08lx",host,PATH_SEP,PATH_SEP,hash);

    if(rw==0)
      {
       int afd=open(file,O_WRONLY|O_CREAT|O_TRUNC,0644);
       if(afd!=-1)
         {
          write(afd,args,strlen(args));
          close(afd);
         }
       else
         {
          close(fd);
          fd=-1;
         }
      }
    else if(rw==1)
      {
       int afd=open(file,O_RDONLY);
       if(afd!=-1)
          close(afd);
       else
         {
          close(fd);
          fd=-1;
         }
      }
    else if(rw==-1)
       unlink(file);
    else if(rw==-2)
       utime(file,NULL);
   }

 free(file);
 free(newpath);

 /* Change the modification time on the directory. */

 if((rw==0 && fd!=-1) || rw==-1 || rw==-2)
    utime(host,NULL);

 return(fd);
}
