/***************************************
  $Header: /home/amb/wwwoffle/RCS/config.c 2.98 2000/11/14 19:14:24 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.6.
  Configuration item management functions.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98,99,2000 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <dirent.h>

#include "misc.h"
#include "configfile.h"
#include "config.h"
#include "proto.h"
#include "sockets.h"
#include "errors.h"
#include "wwwoffle.h"


#ifndef SPOOL_DIR
#define SPOOL_DIR DEF_SPOOL
#endif

#ifndef CONF_DIR
#define CONF_DIR DEF_CONF
#endif

#ifndef PATH_MAX
#define PATH_MAX 4096
#endif

/* StartUp section */

/*+ The port number to use for the HTTP proxy port. +*/
ConfigItem HTTP_Port;

/*+ The port number to use for the wwwoffle port. +*/
ConfigItem WWWOFFLE_Port;

/*+ The spool directory. +*/
ConfigItem SpoolDir;

/*+ The user id for wwwoffled or -1 for none. +*/
ConfigItem WWWOFFLE_Uid;

/*+ The group id for wwwoffled or -1 for none. +*/
ConfigItem WWWOFFLE_Gid;

/*+ Whether to use the syslog facility or not. +*/
ConfigItem UseSyslog;

/*+ The password required for demon configuration. +*/
ConfigItem PassWord;

/*+ Maximum number of servers  +*/
ConfigItem MaxServers;          /*+ in total. +*/
ConfigItem MaxFetchServers;     /*+ for fetching. +*/

/*+ The entries in the StartUp section. +*/
static ConfigEntry startup_entries[]={{"http-port"        ,&HTTP_Port       ,0,Fixed,PortNumber        ,NULL     },  /* 0 see InitConfigurationFile() */
                                      {"wwwoffle-port"    ,&WWWOFFLE_Port   ,0,Fixed,PortNumber        ,NULL     },  /* 1 see InitConfigurationFile() */
                                      {"spool-dir"        ,&SpoolDir        ,0,Fixed,PathName          ,SPOOL_DIR},
                                      {"run-uid"          ,&WWWOFFLE_Uid    ,0,Fixed,UserId            ,"-1"     },
                                      {"run-gid"          ,&WWWOFFLE_Gid    ,0,Fixed,GroupId           ,"-1"     },
                                      {"use-syslog"       ,&UseSyslog       ,0,Fixed,Boolean           ,"true"   },
                                      {"password"         ,&PassWord        ,0,Fixed,String            ,NULL     },
                                      {"max-servers"      ,&MaxServers      ,0,Fixed,CfgMaxServers     ,NULL     },  /* 7 see InitConfigurationFile() */
                                      {"max-fetch-servers",&MaxFetchServers ,0,Fixed,CfgMaxFetchServers,NULL     }}; /* 8 see InitConfigurationFile() */

/*+ The StartUp section. +*/
static ConfigSection startup_section={"StartUp",sizeof(startup_entries)/sizeof(ConfigEntry),startup_entries};


/* Options Section */

/*+ The level of error logging +*/
ConfigItem _LogLevel;

/*+ The amount of time that a socket connection will wait for data. +*/
ConfigItem SocketTimeout;

/*+ The amount of time that a DNS loookup will wait. +*/
ConfigItem DNSTimeout;

/*+ The amount of time that a socket will wait for the intial connection. +*/
ConfigItem ConnectTimeout;

/*+ The option to retry a failed connection. +*/
ConfigItem ConnectRetry;

/*+ The list of allowed SSL port numbers. +*/
ConfigItem SSLAllowPort;

/*+ The permissions for creation of +*/
ConfigItem DirPerm;             /*+ directories. +*/
ConfigItem FilePerm;            /*+ files. +*/

/*+ The name of a progam to run when changing mode to +*/
ConfigItem RunOnline;           /*+ online. +*/
ConfigItem RunOffline;          /*+ offline. +*/
ConfigItem RunAutodial;         /*+ auto dial. +*/
ConfigItem RunFetch;            /*+ fetch (start or stop). +*/

/*+ The option to have lock files to stop some problems. +*/
ConfigItem LockFiles;

/*+ The entries in the Options section. +*/
static ConfigEntry options_entries[]={{"log-level"      ,&_LogLevel     ,0,Fixed,CfgLogLevel,"Important"},
                                      {"socket-timeout" ,&SocketTimeout ,0,Fixed,TimeSecs   ,"120"      },
                                      {"dns-timeout",    &DNSTimeout    ,0,Fixed,TimeSecs   ,"60"       },
                                      {"connect-timeout",&ConnectTimeout,0,Fixed,TimeSecs   ,"30"       },
                                      {"connect-retry"  ,&ConnectRetry  ,0,Fixed,Boolean    ,"no"       },
                                      {"ssl-allow-port" ,&SSLAllowPort  ,0,Fixed,PortNumber ,NULL       },
                                      {"dir-perm"       ,&DirPerm       ,0,Fixed,FileMode   ,NULL       }, /* 6 see InitConfigurationFile() */
                                      {"file-perm"      ,&FilePerm      ,0,Fixed,FileMode   ,NULL       }, /* 7 see InitConfigurationFile() */
                                      {"run-online"     ,&RunOnline     ,0,Fixed,PathName   ,NULL       },
                                      {"run-offline"    ,&RunOffline    ,0,Fixed,PathName   ,NULL       },
                                      {"run-autodial"   ,&RunAutodial   ,0,Fixed,PathName   ,NULL       },
                                      {"run-fetch"      ,&RunFetch      ,0,Fixed,PathName   ,NULL       },
                                      {"lock-files"     ,&LockFiles     ,0,Fixed,Boolean    ,"no"       }};

/*+ Options section. +*/
static ConfigSection options_section={"Options",sizeof(options_entries)/sizeof(ConfigEntry),options_entries};


/* OnlineOptions section. */

/*+ The maximum age of a cached page to use in preference while online. +*/
ConfigItem RequestChanged;

/*+ The option to only request changes to a page once per session online. +*/
ConfigItem RequestChangedOnce;

/*+ The option to re-request pages that have expired. +*/
ConfigItem RequestExpired;

/*+ The option to re-request pages that have the no-cache flag set. +*/
ConfigItem RequestNoCache;

/*+ The option to try and get the requested URL without a password as well as with. +*/
ConfigItem TryWithoutPassword;

/*+ The option to keep downloads that are interrupted by the user. +*/
ConfigItem IntrDownloadKeep;

/*+ The option to keep on downloading interrupted pages if +*/
ConfigItem IntrDownloadSize;           /*+ smaller than a given size. +*/
ConfigItem IntrDownloadPercent;        /*+ more than a given percentage complete. +*/

/*+ The option to keep downloads that time out. +*/
ConfigItem TimeoutDownloadKeep;

/*+ The entries in the OnlineOptions section. +*/
static ConfigEntry onlineoptions_entries[]={{"request-changed"      ,&RequestChanged     ,1,Fixed,TimeSecs  ,"600"},
                                            {"request-changed-once" ,&RequestChangedOnce ,1,Fixed,Boolean   ,"yes"},
                                            {"request-expired"      ,&RequestExpired     ,1,Fixed,Boolean   ,"no" },
                                            {"request-no-cache"     ,&RequestNoCache     ,1,Fixed,Boolean   ,"no" },
                                            {"try-without-password" ,&TryWithoutPassword ,1,Fixed,Boolean   ,"yes"},
                                            {"intr-download-keep"   ,&IntrDownloadKeep   ,1,Fixed,Boolean   ,"no" },
                                            {"intr-download-size"   ,&IntrDownloadSize   ,1,Fixed,FileSize  ,"1"  },
                                            {"intr-download-percent",&IntrDownloadPercent,1,Fixed,Percentage,"80" },
                                            {"timeout-download-keep",&TimeoutDownloadKeep,1,Fixed,Boolean   ,"no" }};

/*+ OnlineOptions section. +*/
static ConfigSection onlineoptions_section={"OnlineOptions",sizeof(onlineoptions_entries)/sizeof(ConfigEntry),onlineoptions_entries};


/* OfflineOptions section. */

/*+ The option to allow or ignore the 'Pragma: no-cache' request. +*/
ConfigItem PragmaNoCache;

/*+ The option to not automatically make requests while offline but to need confirmation. +*/
ConfigItem ConfirmRequests;

/*+ The list of URLs not to request. +*/
ConfigItem DontRequestOffline;

/*+ The entries in the OfflineOptions section. +*/
static ConfigEntry offlineoptions_entries[]={{"pragma-no-cache" ,&PragmaNoCache     ,1,Fixed,Boolean,"yes"},
                                             {"confirm-requests",&ConfirmRequests   ,1,Fixed,Boolean,"no" },
                                             {"dont-request"    ,&DontRequestOffline,1,Fixed,Boolean,"no" }};

/*+ OfflineOptions section. +*/
static ConfigSection offlineoptions_section={"OfflineOptions",sizeof(offlineoptions_entries)/sizeof(ConfigEntry),offlineoptions_entries};


/* FetchOptions section */

/*+ The option to also fetch style sheets. +*/
ConfigItem FetchStyleSheets;

/*+ The option to also fetch images. +*/
ConfigItem FetchImages;

/*+ The option to also fetch frames. +*/
ConfigItem FetchFrames;

/*+ The option to also fetch scripts. +*/
ConfigItem FetchScripts;

/*+ The option to also fetch objects. +*/
ConfigItem FetchObjects;

/*+ The entries in the FetchOptions section. +*/
static ConfigEntry fetchoptions_entries[]={{"stylesheets",&FetchStyleSheets,0,Fixed,Boolean,"no"},
                                           {"images"     ,&FetchImages     ,0,Fixed,Boolean,"no"},
                                           {"frames"     ,&FetchFrames     ,0,Fixed,Boolean,"no"},
                                           {"scripts"    ,&FetchScripts    ,0,Fixed,Boolean,"no"},
                                           {"objects"    ,&FetchObjects    ,0,Fixed,Boolean,"no"}};

/*+ The FetchOptions section. +*/
static ConfigSection fetchoptions_section={"FetchOptions",sizeof(fetchoptions_entries)/sizeof(ConfigEntry),fetchoptions_entries};


/* IndexOptions section */

/*+ The option to disable the lasttime/prevtime indexes. +*/
ConfigItem NoLasttimeIndex;

/*+ The choice of URLs to list in the outgoing index. +*/
ConfigItem IndexListOutgoing;

/*+ The choice of URLs to list in the lastime/prevtime and lastout/prevout indexes. +*/
ConfigItem IndexListLatest;

/*+ The choice of URLs to list in the monitor index. +*/
ConfigItem IndexListMonitor;

/*+ The choice of URLs to list in the host indexes. +*/
ConfigItem IndexListHost;

/*+ The choice of URLs to list in any index. +*/
ConfigItem IndexListAny;

/*+ The entries in the IndexOptions section. +*/
static ConfigEntry indexoptions_entries[]={{"no-lasttime-index",&NoLasttimeIndex  ,0,Fixed,Boolean,"no" },
                                           {"list-outgoing"    ,&IndexListOutgoing,1,Fixed,Boolean,"yes"},
                                           {"list-latest"      ,&IndexListLatest  ,1,Fixed,Boolean,"yes"},
                                           {"list-monitor"     ,&IndexListMonitor ,1,Fixed,Boolean,"yes"},
                                           {"list-host"        ,&IndexListHost    ,1,Fixed,Boolean,"yes"},
                                           {"list-any"         ,&IndexListAny     ,1,Fixed,Boolean,"yes"}};

/*+ The IndexOptions section. +*/
static ConfigSection indexoptions_section={"IndexOptions",sizeof(indexoptions_entries)/sizeof(ConfigEntry),indexoptions_entries};


/* ModifyHTML section */

/*+ The option to turn on the modifications in this section. +*/
ConfigItem EnableHTMLModifications;

/*+ The option to turn on the modifications when online. +*/
ConfigItem EnableModificationsOnline;

/*+ The option of a tag that can be added to the bottom of the spooled pages with the date and some buttons. +*/
ConfigItem AddCacheInfo;

/*+ The options to modify the anchor tags in the HTML. +*/
ConfigItem AnchorModifyBegin[3];
ConfigItem AnchorModifyEnd[3];

/*+ The option to disable scripts and scripted actions. +*/
ConfigItem DisableHTMLScript;

/*+ The option to disable the <blink> tag. +*/
ConfigItem DisableHTMLBlink;

/*+ The option to disable any <meta http-equiv=refresh content=""> tags. +*/
ConfigItem DisableHTMLMetaRefresh;

/*+ The option to disable any <meta http-equiv=refresh content=""> tags that refer to the same URL. +*/
ConfigItem DisableHTMLMetaRefreshSelf;

/*+ The option to demoronise MS characters. +*/
ConfigItem DemoroniseMSChars;

/*+ The option to disable animated GIFs. +*/
ConfigItem DisableAnimatedGIF;

/*+ The entries in the ModifyHTMLOptions section. +*/
static ConfigEntry modifyhtml_entries[]={{"enable-modify-html"       ,&EnableHTMLModifications   ,1,Fixed,Boolean,"no"},
                                         {"enable-modify-online"     ,&EnableModificationsOnline ,1,Fixed,Boolean,"no"},
                                         {"add-cache-info"           ,&AddCacheInfo              ,1,Fixed,Boolean,"no"},
                                         {"anchor-cached-begin"      ,&AnchorModifyBegin[0]      ,1,Fixed,String ,NULL},
                                         {"anchor-cached-end"        ,&AnchorModifyEnd[0]        ,1,Fixed,String ,NULL},
                                         {"anchor-requested-begin"   ,&AnchorModifyBegin[1]      ,1,Fixed,String ,NULL},
                                         {"anchor-requested-end"     ,&AnchorModifyEnd[1]        ,1,Fixed,String ,NULL},
                                         {"anchor-not-cached-begin"  ,&AnchorModifyBegin[2]      ,1,Fixed,String ,NULL},
                                         {"anchor-not-cached-end"    ,&AnchorModifyEnd[2]        ,1,Fixed,String ,NULL},
                                         {"disable-script"           ,&DisableHTMLScript         ,1,Fixed,Boolean,"no"},
                                         {"disable-blink"            ,&DisableHTMLBlink          ,1,Fixed,Boolean,"no"},
                                         {"disable-meta-refresh"     ,&DisableHTMLMetaRefresh    ,1,Fixed,Boolean,"no"},
                                         {"disable-meta-refresh-self",&DisableHTMLMetaRefreshSelf,1,Fixed,Boolean,"no"},
                                         {"demoronise-ms-chars"      ,&DemoroniseMSChars         ,1,Fixed,Boolean,"no"},
                                         {"disable-animated-gif"     ,&DisableAnimatedGIF        ,1,Fixed,Boolean,"no"}};

/*+ The ModifyHTML section. +*/
static ConfigSection modifyhtml_section={"ModifyHTML",sizeof(modifyhtml_entries)/sizeof(ConfigEntry),modifyhtml_entries};


/* LocalHost section */

/*+ The list of localhost hostnames. +*/
ConfigItem LocalHost;

/*+ The entries in the LocalHost section. +*/
static ConfigEntry localhost_entries[]={{""  ,&LocalHost,0,Host,None,NULL}};

/*+ The LocalHost section. +*/
static ConfigSection localhost_section={"LocalHost",sizeof(localhost_entries)/sizeof(ConfigEntry),localhost_entries};


/* LocalNet section */

/*+ The list of local network hostnames. +*/
ConfigItem LocalNet;

/*+ The entries in the LocalNet section. +*/
static ConfigEntry localnet_entries[]={{""  ,&LocalNet,0,Host,None,NULL}};

/*+ The LocalNet section. +*/
static ConfigSection localnet_section={"LocalNet",sizeof(localnet_entries)/sizeof(ConfigEntry),localnet_entries};


/* AllowedConnectHosts section */

/*+ The list of allowed hostnames. +*/
ConfigItem AllowedConnectHosts;

/*+ The entries in the AllowedConnectHosts section. +*/
static ConfigEntry allowedconnecthosts_entries[]={{""  ,&AllowedConnectHosts,0,Host,None,NULL}};

/*+ The AllowedConnectHosts section. +*/
static ConfigSection allowedconnecthosts_section={"AllowedConnectHosts",sizeof(allowedconnecthosts_entries)/sizeof(ConfigEntry),allowedconnecthosts_entries};


/* AllowedConnectUsers section */

/*+ The list of allowed usernames and paswords. +*/
ConfigItem AllowedConnectUsers;

/*+ The entries in the AllowedConnectUsers section. +*/
static ConfigEntry allowedconnectusers_entries[]={{""  ,&AllowedConnectUsers,0,UserPass,None,NULL}};

/*+ The AllowedConnectUsers section. +*/
static ConfigSection allowedconnectusers_section={"AllowedConnectUsers",sizeof(allowedconnectusers_entries)/sizeof(ConfigEntry),allowedconnectusers_entries};


/* DontCache section */

/*+ The list of URLs not to cache. +*/
ConfigItem DontCache;

/*+ The entries in the DontCache section. +*/
static ConfigEntry dontcache_entries[]={{""  ,&DontCache,0,UrlSpecification,None,NULL}};

/*+ The DontCache section. +*/
static ConfigSection dontcache_section={"DontCache",sizeof(dontcache_entries)/sizeof(ConfigEntry),dontcache_entries};


/* DontGet section */

/*+ The list of URLs not to get. +*/
ConfigItem DontGet;

/*+ The replacement URL. +*/
ConfigItem DontGetReplacementURL;

/*+ The list of URLs not to get recursively. +*/
ConfigItem DontGetRecursive;

/*+ The option to treat location headers to not got pages as errors. +*/
ConfigItem DontGetLocation;

/*+ The entries in the DontGet section. +*/
static ConfigEntry dontget_entries[]={{"replacement"   ,&DontGetReplacementURL,1,Fixed           ,Url    ,NULL },
                                      {"get-recursive" ,&DontGetRecursive     ,1,Fixed           ,Boolean,"yes"},
                                      {"location-error",&DontGetLocation      ,1,Fixed           ,Boolean,"no" },
                                      {""              ,&DontGet              ,0,UrlSpecification,None   ,NULL }};

/*+ The DontGet section. +*/
static ConfigSection dontget_section={"DontGet",sizeof(dontget_entries)/sizeof(ConfigEntry),dontget_entries};


/* CensorHeader section */

/*+ The list of censored headers. +*/
ConfigItem CensorHeader;

/*+ Flags to cause the referer header to be mangled. +*/
ConfigItem RefererSelf;
ConfigItem RefererSelfDir;

/*+ The entries in the censor headers section. +*/
static ConfigEntry censorheader_entries[]={{"referer-self"    ,&RefererSelf   ,1,Fixed ,Boolean,"no"},
                                           {"referer-self-dir",&RefererSelfDir,1,Fixed ,Boolean,"no"},
                                           {""                ,&CensorHeader  ,1,String,String ,NULL}};

/*+ The CensorHeader section. +*/
static ConfigSection censorheader_section={"CensorHeader",sizeof(censorheader_entries)/sizeof(ConfigEntry),censorheader_entries};


/* FTPOptions section */

/*+ The anon-ftp username. +*/
ConfigItem FTPUserName;

/*+ The anon-ftp password. +*/
ConfigItem FTPPassWord;

/*+ The information that is needed to allow non-anonymous access, +*/
ConfigItem FTPAuthUser;         /*+ username +*/
ConfigItem FTPAuthPass;         /*+ password +*/

/*+ The entries in the FTPOptions section. +*/
static ConfigEntry ftpoptions_entries[]={{"anon-username",&FTPUserName,0,Fixed,String,"anonymous"},
                                         {"anon-password",&FTPPassWord,0,Fixed,String,NULL       }, /* 2 see InitConfigurationFile() */
                                         {"auth-username",&FTPAuthUser,1,Fixed,String,NULL       },
                                         {"auth-password",&FTPAuthPass,1,Fixed,String,NULL       }};

/*+ The FTPOptions section. +*/
static ConfigSection ftpoptions_section={"FTPOptions",sizeof(ftpoptions_entries)/sizeof(ConfigEntry),ftpoptions_entries};


/* MIMETypes section */

/*+ The default MIME type. +*/
ConfigItem DefaultMIMEType;

/*+ The list of MIME types. +*/
ConfigItem MIMETypes;

/*+ The entries in the FTPOptions section. +*/
static ConfigEntry mimetypes_entries[]={{"default",&DefaultMIMEType,0,Fixed  ,MIMEType,"text/plain"},
                                        {""       ,&MIMETypes      ,0,FileExt,MIMEType,NULL        }};

/*+ The MIMETypes section. +*/
static ConfigSection mimetypes_section={"MIMETypes",sizeof(mimetypes_entries)/sizeof(ConfigEntry),mimetypes_entries};


/* Proxy section */

/*+ The list of hostnames and proxies. +*/
ConfigItem Proxies;

/*+ The information that is needed to allow authorisation headers to be added, +*/
ConfigItem ProxyAuthUser;       /*+ username +*/
ConfigItem ProxyAuthPass;       /*+ password +*/

/*+ The SSL proxy to use. +*/
ConfigItem SSLProxy;

/*+ The entries in the Proxy section. +*/
static ConfigEntry proxy_entries[]={{"auth-username",&ProxyAuthUser,1,Fixed,String           ,NULL},
                                    {"auth-password",&ProxyAuthPass,1,Fixed,String           ,NULL},
                                    {"ssl"          ,&SSLProxy     ,1,Fixed,HostAndPortOrNone,NULL},
                                    {"proxy"        ,&Proxies      ,1,Fixed,HostAndPortOrNone,NULL}};

/*+ The Proxy section. +*/
static ConfigSection proxy_section={"Proxy",sizeof(proxy_entries)/sizeof(ConfigEntry),proxy_entries};


/* Alias section */

/*+ The list of protocols/hostnames and their aliases. +*/
ConfigItem Aliases;

/*+ The entries in the Alias section. +*/
static ConfigEntry alias_entries[]={{"",&Aliases,0,UrlSpecification,UrlSpecification,NULL}};

/*+ The Alias section. +*/
static ConfigSection alias_section={"Alias",sizeof(alias_entries)/sizeof(ConfigEntry),alias_entries};


/* Purge section */

/*+ A flag to indicate that the modification time is used instead of the access time. +*/
ConfigItem PurgeUseMTime;

/*+ The maximum allowed size of the cache. +*/
ConfigItem PurgeCacheSize;

/*+ The minimum allowed free disk space. +*/
ConfigItem PurgeDiskFree;

/*+ A flag to indicate if the whole URL is used to choose the purge age. +*/
ConfigItem PurgeUseURL;

/*+ A flag to indicate if the DontGet hosts are to be purged. +*/
ConfigItem PurgeDontGet;

/*+ A flag to indicate if the DontCache hosts are to be purged. +*/
ConfigItem PurgeDontCache;

/*+ The list of hostnames and purge ages. +*/
ConfigItem PurgeAges;

/*+ The entries in the Purge section. +*/
static ConfigEntry purge_entries[]={{"use-mtime"     ,&PurgeUseMTime  ,0,Fixed,Boolean  ,"no"},
                                    {"age"           ,&PurgeAges      ,1,Fixed,AgeDays  ,"14"},
                                    {"max-size"      ,&PurgeCacheSize ,0,Fixed,CacheSize,"0" },
                                    {"min-free"      ,&PurgeDiskFree  ,0,Fixed,CacheSize,"0" },
                                    {"use-url"       ,&PurgeUseURL    ,0,Fixed,Boolean  ,"no"},
                                    {"del-dontget"   ,&PurgeDontGet   ,0,Fixed,Boolean  ,"no"},
                                    {"del-dontcache" ,&PurgeDontCache ,0,Fixed,Boolean  ,"no"}};

/*+ The Purge section. +*/
static ConfigSection purge_section={"Purge",sizeof(purge_entries)/sizeof(ConfigEntry),purge_entries};


/* Whole file */

/*+ The list of sections (NULL terminated). +*/
static ConfigSection *file_sections[]={&startup_section,
                                       &options_section,
                                       &onlineoptions_section,
                                       &offlineoptions_section,
                                       &fetchoptions_section,
                                       &indexoptions_section,
                                       &modifyhtml_section,
                                       &localhost_section,
                                       &localnet_section,
                                       &allowedconnecthosts_section,
                                       &allowedconnectusers_section,
                                       &dontcache_section,
                                       &dontget_section,
                                       &censorheader_section,
                                       &ftpoptions_section,
                                       &mimetypes_section,
                                       &proxy_section,
                                       &alias_section,
                                       &purge_section};

/*+ The whole file +*/
static ConfigFile config_file={CONF_DIR "/wwwoffle.conf",sizeof(file_sections)/sizeof(ConfigSection*),file_sections};


/* Local functions */

static char *DefaultFTPPassWord(void);


/*++++++++++++++++++++++++++++++++++++++
  Return the name of the configuration file.

  char *ConfigurationFileName Returns the filename.
  ++++++++++++++++++++++++++++++++++++++*/

char *ConfigurationFileName(void)
{
 return(config_file.name);
}


/*++++++++++++++++++++++++++++++++++++++
  Initialise the configuration file information to the default values.
  ++++++++++++++++++++++++++++++++++++++*/

void InitConfigurationFile(char *name)
{
 static char startup[6][8];

 if(name)
    config_file.name=name;

 if(*config_file.name!='/')
   {
    static char cwd[PATH_MAX+1];

    if(getcwd(cwd,PATH_MAX))
      {
       strcat(cwd,"/");
       strcat(cwd,config_file.name);

       config_file.name=cwd;
      }
   }

 /* Entries that cannot be set at compile time. */

 sprintf(startup[0],"%d",DEF_HTTP_PORT);         startup_entries[0].defval=startup[0];
 sprintf(startup[1],"%d",DEF_WWWOFFLE_PORT);     startup_entries[1].defval=startup[1];
 sprintf(startup[2],"%d",DEF_MAX_SERVERS);       startup_entries[7].defval=startup[2];
 sprintf(startup[3],"%d",DEF_MAX_FETCH_SERVERS); startup_entries[8].defval=startup[3];

 sprintf(startup[4],"0%o",DEF_DIR_PERM);         options_entries[6].defval=startup[4];
 sprintf(startup[5],"0%o",DEF_FILE_PERM);        options_entries[7].defval=startup[5];

 ftpoptions_entries[1].defval=DefaultFTPPassWord();

 /* Convert the default values. */

 DefaultConfigFile(&config_file);

 LogLevel=ConfigInteger(_LogLevel);

 SetDNSTimeout(ConfigInteger(DNSTimeout));
 SetConnectTimeout(ConfigInteger(ConnectTimeout));
 set_read_timeout(ConfigInteger(SocketTimeout));
}


/*++++++++++++++++++++++++++++++++++++++
  Determine an email address to use as the FTP password.

  char *DefaultFTPPassword Returns a best-guess password.
  ++++++++++++++++++++++++++++++++++++++*/

static char *DefaultFTPPassWord(void)
{
 struct passwd *pwd;
 char *username,*fqdn,*password;

 pwd=getpwuid(getuid());

 if(!pwd)
    username="root";
 else
    username=pwd->pw_name;

 fqdn=GetFQDN();

 if(!fqdn)
    fqdn="";

 password=(char*)malloc(strlen(username)+strlen(fqdn)+4);
 sprintf(password,"%s@%s",username,fqdn);

 return(password);
}


/*++++++++++++++++++++++++++++++++++++++
  Read in the configuration file.

  int fd The file descriptor to write the error message to.
  ++++++++++++++++++++++++++++++++++++++*/

int ReadConfigurationFile(int fd)
{
 char *errmsg;

 errmsg=ReadConfigFile(&config_file);

 if(errmsg)
   {
    if(fd!=-1)
       write_string(fd,errmsg);
    free(errmsg);

    return(1);
   }
 else
   {
    LogLevel=ConfigInteger(_LogLevel);

    SetDNSTimeout(ConfigInteger(DNSTimeout));
    SetConnectTimeout(ConfigInteger(ConnectTimeout));
    set_read_timeout(ConfigInteger(SocketTimeout));

    return(0);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Decide if the specified host and port number is allowed for SSL.

  int IsSSLAllowedPort Returns true if it is allowed.

  char *host The hostname and port number to check.
  ++++++++++++++++++++++++++++++++++++++*/

int IsSSLAllowedPort(char *host)
{
 char *colon=strchr(host,':');
 int port,isit=0;
 int i;

 if(!colon)
    return(0);

 port=atoi(colon+1);

 if(SSLAllowPort)
    for(i=0;i<SSLAllowPort->nitems;i++)
       if(SSLAllowPort->val[i].integer==port)
         {isit=1;break;}

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Get the first specified name of the server host.

  char *GetLocalHost Returns the first named localhost.

  int port If true then return the port as well.
  ++++++++++++++++++++++++++++++++++++++*/

char *GetLocalHost(int port)
{
 char *localhost,*ret;

 if(LocalHost && LocalHost->nitems)
    localhost=LocalHost->key[0].string;
 else
    localhost="localhost";

 ret=(char*)malloc(strlen(localhost)+8);

 if(port)
    sprintf(ret,"%s:%d",localhost,ConfigInteger(HTTP_Port));
 else
    strcpy(ret,localhost);

 return(ret);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is the localhost.

  int IsLocalHost Return true if the host is the local host.

  char *host The name of the host (and port number) to be checked.

  int port If true then check the port number as well.
  ++++++++++++++++++++++++++++++++++++++*/

int IsLocalHost(char *host,int port)
{
 char *colon=strchr(host,':');
 int isit=0;
 int i;

 if(colon)
    *colon=0;

 if(LocalHost)
    for(i=0;i<LocalHost->nitems;i++)
       if(!strcmp(LocalHost->key[i].string,host))
         {isit=1;break;}

 if(colon)
    *colon=':';

 if(isit && port)
   {
    if((colon && atoi(colon+1)==ConfigInteger(HTTP_Port)) ||
       (!colon && ConfigInteger(HTTP_Port)==80))
       ;
    else
       isit=0;
   }

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is in the local network.

  int IsLocalNetHost Return true if the host is on the local network.

  char *host The name of the host (and port number) to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsLocalNetHost(char *host)
{
 char *colon=strchr(host,':');
 int isit=0;
 int i;

 if(colon)
    *colon=0;

 if(IsLocalHost(host,0))
    isit=1;

 if(LocalNet && !isit)
   {
    for(i=0;i<LocalNet->nitems;i++)
       if(*LocalNet->key[i].string=='!')
         {
          if(WildcardMatch(host,LocalNet->key[i].string+1))
            {isit=0;break;}
         }
       else
         {
          if(WildcardMatch(host,LocalNet->key[i].string))
            {isit=1;break;}
         }
   }

 if(colon)
    *colon=':';

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is allowed to connect.

  int IsAllowedConnectHost Return true if it is allowed to connect.

  char *host The name of the host to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsAllowedConnectHost(char *host)
{
 int isit=0;
 int i;

 if(LocalHost)
    for(i=0;i<LocalHost->nitems;i++)
       if(!strcmp(LocalHost->key[i].string,host))
         {isit=1;break;}

 if(AllowedConnectHosts && !isit)
   {
    for(i=0;i<AllowedConnectHosts->nitems;i++)
       if(*AllowedConnectHosts->key[i].string=='!')
         {
          if(WildcardMatch(host,AllowedConnectHosts->key[i].string+1))
            {isit=0;break;}
         }
       else
         {
          if(WildcardMatch(host,AllowedConnectHosts->key[i].string))
            {isit=1;break;}
         }
   }

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified username and password is allowed to connect.

  char *IsAllowedConnectUser Return the username if it is allowed to connect.

  char *userpass The encoded username and password of the user to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

char *IsAllowedConnectUser(char *userpass)
{
 char *isit;
 int i;

 if(AllowedConnectUsers)
    isit=NULL;
 else
    isit="anybody";

 if(AllowedConnectUsers && userpass)
   {
    char *up=userpass;

    while(*up!=' ') up++;
    while(*up==' ') up++;

    for(i=0;i<AllowedConnectUsers->nitems;i++)
       if(!strcmp(AllowedConnectUsers->key[i].string,up))
         {
          char *colon;
          int l;
          isit=Base64Decode(AllowedConnectUsers->key[i].string,&l);
          colon=strchr(isit,':');
          *colon=0;
          break;
         }
   }

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Decide if the header line is to be sent to the server/browser.

  char *CensoredHeader Returns the value to be inserted or NULL if it is to be removed.

  URL *Url the URL that the request or reply is for/from.

  char *key The key to check.

  char *val The default value to use.
  ++++++++++++++++++++++++++++++++++++++*/

char *CensoredHeader(URL *Url,char *key,char *val)
{
 char *new=val;
 int i;

 if(CensorHeader)
    for(i=0;i<CensorHeader->nitems;i++)
       if(!strcmp(CensorHeader->key[i].string,key))
          if(!CensorHeader->url[i] || MatchUrlSpecification(CensorHeader->url[i],Url->proto,Url->host,Url->path,Url->args))
            {
             if(!CensorHeader->val[i].string)
                new=NULL;
             else if(!strcmp(CensorHeader->val[i].string,"yes"))
                new=NULL;
             else if(!strcmp(CensorHeader->val[i].string,"no"))
                ;
             else if(strcmp(CensorHeader->val[i].string,val))
               {
                new=(char*)malloc(strlen(CensorHeader->val[i].string)+1);
                strcpy(new,CensorHeader->val[i].string);
               }
             break;
            }

 if(new && !strcmp("Referer",key) &&(ConfigBooleanURL(RefererSelf,Url) || ConfigBooleanURL(RefererSelfDir,Url)))
   {
    new=(char*)malloc(strlen(Url->name)+1);
    strcpy(new,Url->name);
    if(RefererSelfDir)
      {
       char *p=new+strlen(new)-1,*ques=strchr(new,'?');

       if(ques)
          p=ques;

       while(*p!='/')
          *p--=0;
      }
   }

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Decide what mime type to apply for a given file.

  char *WhatMIMEType Returns the MIME Type.

  char *path The path of the file.
  ++++++++++++++++++++++++++++++++++++++*/

char *WhatMIMEType(char *path)
{
 char *mimetype=ConfigString(DefaultMIMEType);
 int maxlen=0;
 int i;

 if(MIMETypes)
    for(i=0;i<MIMETypes->nitems;i++)
       if(strlen(path)>strlen(MIMETypes->key[i].string) &&
          strlen(MIMETypes->key[i].string)>maxlen &&
          !strcmp(MIMETypes->key[i].string,path+strlen(path)-strlen(MIMETypes->key[i].string)))
         {mimetype=MIMETypes->val[i].string;maxlen=strlen(mimetype);}

 return(mimetype);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if a specified protocol and host is aliased to another host.

  int IsAliased Returns non-zero if this is host is aliased to another host.

  char *proto The protocol to check.

  char *host The hostname to check.

  char *path The pathname to check.

  char **new_proto The protocol of the alias.

  char **new_host The hostname of the alias.

  char **new_path The pathname of the alias.
  ++++++++++++++++++++++++++++++++++++++*/

int IsAliased(char *proto,char *host,char *path,char **new_proto,char **new_host,char **new_path)
{
 int i;

 *new_proto=*new_host=*new_path=NULL;

 if(Aliases)
    for(i=0;i<Aliases->nitems;i++)
       if(MatchUrlSpecification(Aliases->key[i].urlspec,proto,host,path,NULL))
         {
          if(Aliases->val[i].urlspec->proto)
            {
             *new_proto=(char*)malloc(strlen(Aliases->val[i].urlspec->proto)+1);
             strcpy(*new_proto,Aliases->val[i].urlspec->proto);
            }
          else
            {
             *new_proto=(char*)malloc(strlen(proto)+1);
             strcpy(*new_proto,proto);
            }

          if(Aliases->val[i].urlspec->host)
            {
             *new_host=(char*)malloc(strlen(Aliases->val[i].urlspec->host)+8);
             strcpy(*new_host,Aliases->val[i].urlspec->host);
             if(Aliases->val[i].urlspec->port>0)
                sprintf((*new_host)+strlen(*new_host),":%d",Aliases->val[i].urlspec->port);
            }
          else
            {
             *new_host=(char*)malloc(strlen(host)+1);
             strcpy(*new_host,host);
            }

          if(Aliases->val[i].urlspec->path)
            {
             int oldlen=Aliases->key[i].urlspec->path?strlen(Aliases->key[i].urlspec->path):0;
             int newlen=Aliases->val[i].urlspec->path?strlen(Aliases->val[i].urlspec->path):0;

             *new_path=(char*)malloc(newlen-oldlen+strlen(path)+1);
             if(newlen)
               {
                strcpy(*new_path,Aliases->val[i].urlspec->path);
                if((*new_path)[newlen-1]=='/')
                   (*new_path)[newlen-1]=0;
               }
             strcat(*new_path,path+oldlen);
            }
          else
            {
             *new_path=(char*)malloc(strlen(path)+1);
             strcpy(*new_path,path);
            }
         }

 return(!!*new_proto);
}


/*++++++++++++++++++++++++++++++++++++++
  Search through a ConfigItem to find an integer value that applies to the specified URL.

  int ConfigInteger Returns the integer value.

  ConfigItem item The configuration item to check.
  ++++++++++++++++++++++++++++++++++++++*/

int ConfigInteger(ConfigItem item)
{
#if CONFIG_VERIFY_ABORT
 if(!item)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);

 if(item->entry->url_type!=0)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);

 if(item->nitems>1)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);
#endif

 if(item->nitems==1)
    return(item->val[0].integer);
 else
    return(item->defval->integer);
}


/*++++++++++++++++++++++++++++++++++++++
  Search through a ConfigItem to find an string value that applies to the specified URL.

  char *ConfigString Returns the string value.

  ConfigItem item The configuration item to check.
  ++++++++++++++++++++++++++++++++++++++*/

char *ConfigString(ConfigItem item)
{
 if(!item)
    return(NULL);

#if CONFIG_VERIFY_ABORT
 if(item->entry->url_type!=0)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);

 if(item->nitems>1)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);
#endif

 if(item->nitems==1)
    return(item->val[0].string);
 else
    return(item->defval->string);
}


/*++++++++++++++++++++++++++++++++++++++
  Search through a ConfigItem to find an integer value that applies to the specified URL.

  int ConfigIntegerURL Returns the integer value.

  ConfigItem item The configuration item to check.

  URL *Url the URL to check for a match against.
  ++++++++++++++++++++++++++++++++++++++*/

int ConfigIntegerURL(ConfigItem item,URL *Url)
{
 int i;

#if CONFIG_VERIFY_ABORT
 if(!item)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);

 if(item->entry->url_type!=1)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);
#endif

 for(i=0;i<item->nitems;i++)
   {
    if(!item->url[i])
       return(item->val[i].integer);
    else if(MatchUrlSpecification(item->url[i],Url->proto,Url->host,Url->path,Url->args))
       return(item->val[i].integer);
   }

 return(item->defval->integer);
}


/*++++++++++++++++++++++++++++++++++++++
  Search through a ConfigItem to find an string value that applies to the specified URL.

  char *ConfigStringURL Returns the string value.

  ConfigItem item The configuration item to check.

  URL *Url the URL to check for a match against.
  ++++++++++++++++++++++++++++++++++++++*/

char *ConfigStringURL(ConfigItem item,URL *Url)
{
 int i;

 if(!item)
    return(NULL);

#if CONFIG_VERIFY_ABORT
 if(item->entry->url_type!=1)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);
#endif

 for(i=0;i<item->nitems;i++)
   {
    if(!item->url[i])
       return(item->val[i].string);
    else if(MatchUrlSpecification(item->url[i],Url->proto,Url->host,Url->path,Url->args))
       return(item->val[i].string);
   }

 if(item->defval)
    return(item->defval->string);
 else
    return(NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified string is listed in this configuration item.

  char *ConfigStringMatchString Return the string associated with the specified string.

  char *string The string to search the list for.
  ++++++++++++++++++++++++++++++++++++++*/

char *ConfigStringMatchString(ConfigItem item,char *string)
{
 int i;

 if(!item)
    return(NULL);

#if CONFIG_VERIFY_ABORT
 if(item->entry->url_type!=0)
    PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);
#endif

 for(i=0;i<item->nitems;i++)
    if(!strcmp(item->key[i].string,string))
       return(item->val[i].string);

 return(NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified URL is listed in this configuration item.

  int ConfigBooleanMatchURL Return true if it is in the list.

  URL *Url The URL to search the list for.
  ++++++++++++++++++++++++++++++++++++++*/

int ConfigBooleanMatchURL(ConfigItem item,URL *Url)
{
 int i;

 if(item)
    for(i=0;i<item->nitems;i++)
       if(MatchUrlSpecification(item->key[i].urlspec,Url->proto,Url->host,Url->path,Url->args))
          return(!item->key[i].urlspec->negated);

 return(0);
}
